#include "Arduino.h"
#include "DS3231lib.h"
#include <Wire.h>

//Constructor
DS3231::DS3231(bool I2Cinit, byte I2Caddress) {
  _I2Caddress = I2Caddress;
  if (I2Cinit)
    Wire.begin();
}

//Decimal to Binary Coded Decimal
byte DS3231::DECtoBCD(byte val) {
  return (((val / 10) << 4) + val % 10);
}

//Binary Coded Decimal to Decimal
byte DS3231::BCDtoDEC(byte val) {
  return (10 * (val >> 4) + val % 16);
}

//Change the time registers of the RTC
bool DS3231::setTime(byte Sec, byte Min, byte Hour, byte Day, byte Date, byte Month, byte Year) {
  if ((0 <= Sec) && (Sec <= 59) &&
      (0 <= Min) && (Min <= 59) &&
      (0 <= Hour) && (Hour <= 23) &&
      (1 <= Day) && (Day <= 7) &&
      (1 <= Date) && (Date <= 31) &&
      (1 <= Month) && (Month <= 12) &&
      (0 <= Year) && (Year <= 99))
  {
    Wire.beginTransmission(_I2Caddress);
    Wire.write(0x00);
    Wire.write(DECtoBCD(Sec));
    Wire.write(DECtoBCD(Min));
    Wire.write(DECtoBCD(Hour));
    Wire.write(DECtoBCD(Day));
    Wire.write(DECtoBCD(Date));
    Wire.write(DECtoBCD(Month));
    Wire.write(DECtoBCD(Year));
    Wire.endTransmission();
    return true;
  }
  else
    return false;
}

//Read the time registers of the RTC
bool DS3231::getTime(Time* t) {
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x00);
  Wire.endTransmission();

  Wire.requestFrom(_I2Caddress, 7);
  (*t).Seconds = BCDtoDEC(Wire.read());
  (*t).Minutes = BCDtoDEC(Wire.read());
  (*t).Hours = BCDtoDEC(Wire.read());
  (*t).Day = BCDtoDEC(Wire.read());
  (*t).Date = BCDtoDEC(Wire.read());
  (*t).Month = BCDtoDEC(Wire.read());
  (*t).Year = BCDtoDEC(Wire.read());
  return true;
}

//Read the Time and Display it in the Serial Monitor
bool DS3231::displayTimeSerial() {
  Time t;
  String DayOfWeek;
  getTime(&t);
  if (t.Hours < 10) Serial.print("0");
  Serial.print(t.Hours); Serial.print(":");
  if (t.Minutes < 10) Serial.print("0");
  Serial.print(t.Minutes); Serial.print(":");
  if (t.Seconds < 10) Serial.print("0");
  Serial.print(t.Seconds); Serial.print("  ");
  switch (t.Day) {
    case 1: DayOfWeek = "Monday"; break;
    case 2: DayOfWeek = "Tuesday"; break;
    case 3: DayOfWeek = "Wednesday"; break;
    case 4: DayOfWeek = "Thursday"; break;
    case 5: DayOfWeek = "Friday"; break;
    case 6: DayOfWeek = "Saturday"; break;
    case 7: DayOfWeek = "Sunday"; break;
  }
  Serial.print(DayOfWeek + ", ");
  if (t.Date < 10) Serial.print("0");
  Serial.print(t.Date); Serial.print(".");
  if (t.Month < 10) Serial.print("0");
  Serial.print(t.Month); Serial.print(".");
  Serial.println(t.Year);
  return true;
}

//Set Alarm 1
bool DS3231::setAlarm1(byte Sec, byte Min, byte Hour, byte DayDate, byte Setting) {
  if (bitRead(_Alarm1Mask[Setting], 7)) {
    if ((1 <= DayDate) && (DayDate <= 7));
    else return false;
  }
  if ((0 <= Sec) && (Sec <= 59) &&
      (0 <= Min) && (Min <= 59) &&
      (0 <= Hour) && (Hour <= 23) &&
      (1 <= DayDate) && (DayDate <= 31) &&
      (0 <= Setting) && (Setting <= 5))
  {
    Sec = DECtoBCD(Sec) + bitRead(_Alarm1Mask[Setting], 3) * 128;
    Min = DECtoBCD(Min) + bitRead(_Alarm1Mask[Setting], 4) * 128;
    Hour = DECtoBCD(Hour) + bitRead(_Alarm1Mask[Setting], 5) * 128;
    DayDate = DECtoBCD(DayDate) + bitRead(_Alarm1Mask[Setting], 6) * 128 + bitRead(_Alarm1Mask[Setting], 7) * 64;

    Wire.beginTransmission(_I2Caddress);
    Wire.write(0x07);
    Wire.write(Sec);
    Wire.write(Min);
    Wire.write(Hour);
    Wire.write(DayDate);
    Wire.endTransmission();
    return true;
  }
  else
    return false;
}

//Set Alarm 2
bool DS3231::setAlarm2(byte Min, byte Hour, byte DayDate, byte Setting) {
  if (bitRead(_Alarm2Mask[Setting], 7)) {
    if ((1 <= DayDate) && (DayDate <= 7));
    else return false;
  }
  if ((0 <= Min) && (Min <= 59) &&
      (0 <= Hour) && (Hour <= 23) &&
      (1 <= DayDate) && (DayDate <= 31) &&
      (0 <= Setting) && (Setting <= 4))
  {
    Min = DECtoBCD(Min) + bitRead(_Alarm2Mask[Setting], 4) * 128;
    Hour = DECtoBCD(Hour) + bitRead(_Alarm2Mask[Setting], 5) * 128;
    DayDate = DECtoBCD(DayDate) + bitRead(_Alarm2Mask[Setting], 6) * 128 + bitRead(_Alarm2Mask[Setting], 7) * 64;

    Wire.beginTransmission(_I2Caddress);
    Wire.write(0x0B);
    Wire.write(Min);
    Wire.write(Hour);
    Wire.write(DayDate);
    Wire.endTransmission();
    return true;
  }
  else
    return false;
}

/*==============================================================
   Status Register
  ==============================================================*/

//Get status register
byte DS3231::getStatus() {
  byte Status;
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0F);
  Wire.endTransmission();
  Wire.requestFrom(_I2Caddress, 1);
  Status = Wire.read();
  return Status;
}

bool DS3231::displayStatusSerial() {
  byte Status = getStatus();
  if (bitRead(Status, 7))
    Serial.println("Oscillator has stopped");
  else Serial.println("Oscillator is running.");
  if (bitRead(Status, 3))
    Serial.println("32kHz Output is enabled.");
  else Serial.println("32kHz Output is disabled.");
  if (bitRead(Status, 2))
    Serial.println("RTC is busy: Temperature Conversion.");
  else Serial.println("RTC is idle: No conversion in process.");
  if (bitRead(Status, 1))
    Serial.println("Alarm 2 was triggered.");
  else Serial.println("Alarm 2 was not triggered.");
  if (bitRead(Status, 0))
    Serial.println("Alarm 1 was triggered.");
  else Serial.println("Alarm 1 was not triggered.");
  return true;
}

//Reset alarm 1
bool DS3231::resetAlarm1() {
  byte Status = getStatus();
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0F);
  Wire.write(bitWrite(Status, 0, 0));
  Wire.endTransmission();
}

//Reset alarm 2
bool DS3231::resetAlarm2() {
  byte Status = getStatus();
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0F);
  Wire.write(bitWrite(Status, 1, 0));
  Wire.endTransmission();
}

//Check the OSF (Oscillator Stop Flag), returns flag value
bool DS3231::getOSF() {
  byte Status = getStatus();
  return bitRead(Status, 7);
}

//Reset Oscillator Stop Flag (OSF)
bool DS3231::resetOSF() {
  byte Status = getStatus();
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0F);
  Wire.write(bitWrite(Status, 7, 0));
  Wire.endTransmission();
  return true;
}

//Check if SQW signal output is activated (EN32kHz)
bool DS3231::getEN32kHz() {
  byte Status = getStatus();
  return bitRead(Status, 3);
}

//(De)activate SQW signal output (EN32kHz)
bool DS3231::setEN32kHz(bool active) {
  byte Status = getStatus();
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0F);
  Wire.write(bitWrite(Status, 3, active));
  Wire.endTransmission();
}

//Check if RTC is busy (BSY)
bool DS3231::getBSY() {
  byte Status = getStatus();
  return bitRead(Status, 2);
}

//Check alarm 1 flag (A1F)
bool DS3231::getA1F() {
  byte Status = getStatus();
  return bitRead(Status, 0);
}

//Check alarm 1 flag (A2F)
bool DS3231::getA2F() {
  byte Status = getStatus();
  return bitRead(Status, 1);
}

/*==============================================================
   Control Register
  ==============================================================*/

byte DS3231::getControl() {
  byte Control;
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0E);
  Wire.endTransmission();
  Wire.requestFrom(_I2Caddress, 1);
  Control = Wire.read();
  return Control;
}

bool DS3231::displayControlSerial(){
  byte Control = getControl();
  if(bitRead(Control, 7))
    Serial.println("Oscillator is disabled.");
  else Serial.println("Oscillator is enabled");
  if(bitRead(Control, 6))
    Serial.println("Battery-Backed Square-Wave Enable is activated");
  else Serial.println("Battery-Backed Square-Wave Enable is deactivated.");
  if(bitRead(Control, 5))
    Serial.println("Temperature conversion is issued");
  else Serial.println("No temperature conversion is issued.");
  Serial.print("Square wave output frequency is set to ");
  byte Setting = getSQWOF();
  switch (Setting) {
    case 0: Serial.println("1Hz"); break;
    case 1: Serial.println("1.024kHz"); break;
    case 2: Serial.println("4.096kHz"); break;
    case 3: Serial.println("8.192kHz"); break;
  }
  if(bitRead(Control, 2))
    Serial.println("Interrupts are activated.");
  else Serial.println("Interrupts are deactivated.");
  if(bitRead(Control, 1))
    Serial.println("Alarm 2 is enabled.");
  else Serial.println("Alarm 2 is disabled.");
  if(bitRead(Control, 0))
    Serial.println("Alarm 1 is enabled.");
  else Serial.println("Alarm 1 is disabled.");
}

bool DS3231::setControl(byte Setting) {
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0E);
  Wire.write(Setting);
  Wire.endTransmission();
  return true;
}

bool DS3231::getEOSC() {
  byte Control = getControl();
  return bitRead(Control, 7);
}

bool DS3231::setEOSC(bool active) {
  byte Control = getControl();
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0E);
  Wire.write(bitWrite(Control, 7, active));
  Wire.endTransmission();
  return true;
}

bool DS3231::getBBSQW() {
  byte Control = getControl();
  return bitRead(Control, 6);
}

bool DS3231::setBBSQW(bool active) {
  byte Control = getControl();
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0E);
  Wire.write(bitWrite(Control, 6, active));
  Wire.endTransmission();
  return true;
}

byte DS3231::getSQWOF() {
  byte Control = getControl();
  byte Setting = bitRead(Control, 4) * 2 + bitRead(Control, 3);
  return Setting;
}

bool DS3231::setSQWOF(byte f) {
  if ( (0 <= f) && (f <= 3)){
    byte Control = getControl();
    bitWrite(Control, 4, bitRead(f,1));
    bitWrite(Control, 3, bitRead(f,0));
    Wire.beginTransmission(_I2Caddress);
    Wire.write(0x0E);
    Wire.write(Control);
    Wire.endTransmission();
    return true;
  }
  else return false;
}

bool DS3231::getINTCN() {
  byte Control = getControl();
  return bitRead(Control, 2);
}

bool DS3231::setINTCN(bool active) {
  byte Control = getControl();
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0E);
  Wire.write(bitWrite(Control, 2, active));
  Wire.endTransmission();
  return true;
}

bool DS3231::getA2IE() {
  byte Control = getControl();
  return bitRead(Control, 1);
}

bool DS3231::setA2IE(bool active) {
  byte Control = getControl();
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0E);
  Wire.write(bitWrite(Control, 1, active));
  Wire.endTransmission();
  return true;
}

bool DS3231::getA1IE() {
  byte Control = getControl();
  return bitRead(Control, 0);
}

bool DS3231::setA1IE(bool active) {
  byte Control = getControl();
  Wire.beginTransmission(_I2Caddress);
  Wire.write(0x0E);
  Wire.write(bitWrite(Control, 0, active));
  Wire.endTransmission();
  return true;
}
